<?php
namespace App\Livewire\Payment;

use Livewire\Component;
use App\Models\Payment;
use Illuminate\Support\Carbon;

class MonthlyPaymentReport extends Component
{
    public $selectedMonth;
    public $selectedYear;
    public $dueFilter = 'all'; // all, cleared, not-cleared
    public $reportType = 'monthly'; // monthly, quarterly, yearly

    public function mount()
    {
        $now = Carbon::now();
        $this->selectedMonth = $now->month;
        $this->selectedYear = $now->year;
    }

    public function fetchPayments()
    {
        $query = Payment::query();

        // Apply due filter
        if ($this->dueFilter === 'not-cleared') {
            $query->where('due', 'not-cleared');
        } elseif ($this->dueFilter === 'cleared') {
            $query->where('due', 'cleared');
        }

        // Apply the report type logic (monthly, quarterly, yearly)
        if ($this->reportType === 'monthly') {
            $query->selectRaw('
                YEAR(created_at) as year,
                MONTH(created_at) as month,
                SUM(total) as total_amount,
                SUM(advance) as total_advance,
                SUM(CASE WHEN due = "not-cleared" THEN (total-advance) ELSE 0 END) as pending_amount
            ')
            ->whereYear('created_at', $this->selectedYear)
            ->whereMonth('created_at', $this->selectedMonth)
            ->groupBy('year', 'month')
            ->orderBy('year', 'desc')
            ->orderBy('month', 'desc');
        } elseif ($this->reportType === 'quarterly') {
            $query->selectRaw('
                YEAR(created_at) as year,
                QUARTER(created_at) as quarter,
                SUM(total) as total_amount,
                SUM(advance) as total_advance,
                SUM(CASE WHEN due = "not-cleared" THEN total ELSE 0 END) as pending_amount
            ')
            ->whereYear('created_at', $this->selectedYear)
            ->groupBy('year', 'quarter')
            ->orderBy('year', 'desc')
            ->orderBy('quarter', 'asc');
        } else {
            // For the yearly report, group by year and show total for all years
            $query->selectRaw('
                YEAR(created_at) as year,
                SUM(total) as total_amount,
                SUM(advance) as total_advance,
                SUM(CASE WHEN due = "not-cleared" THEN total ELSE 0 END) as pending_amount
            ')
            ->whereYear('created_at', $this->selectedYear)
            ->groupBy('year')
            ->orderBy('year', 'desc');
        }

        // If no month, quarter or year filter is selected, show all entries for the selected year
        if ($this->reportType === 'yearly' && $this->selectedMonth == null && $this->dueFilter === 'all') {
            $query->selectRaw('
                YEAR(created_at) as year,
                SUM(total) as total_amount,
                SUM(advance) as total_advance,
                SUM(CASE WHEN due = "not-cleared" THEN total ELSE 0 END) as pending_amount
            ')
            ->groupBy('year')
            ->orderBy('year', 'desc');
        }

        return $query->get();
    }

    public function render()
    {
        $payments = $this->fetchPayments();

        return view('livewire.payment.monthly-payment-report', compact('payments'));
    }
}
